/*
	Description: Constructs Table A4 in the online appendix

	Author: Brendan Price
	Prepared for public disclosure: 4/28/14
*/

version 12.1
clear all
set more off
set matsize 11000
set trace off
capture restore, not
capture log close
pause on

/* Construct employment weights: industry's average share of manufacturing employment over the sample period */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep sic87dd year emp
keep if year >= 1977 & year <= 2007
bysort year: egen totemp = total(emp)
gen empsh = emp/totemp
bysort sic87dd: egen wt = mean(empsh)
keep sic87dd wt
duplicates drop
bysort sic87dd: assert _N == 1
egen totwt = total(wt)
assert abs(totwt - 1) < .001
keep sic87dd wt
tempfile weights
save "`weights'.dta", replace

/* Prepare data on computer investment rates */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep if year == 1977 | year == 1982 | year == 1987 | year == 1992 | year == 2002 | year == 2007
keep sic87dd year comp_broad nom_compinvest* nom_invest* real_vship nom_vship emp real_pay real_prodpay
rename real_pay pay
rename real_prodpay prodpay
reshape wide nom_invest real_vship nom_vship emp pay prodpay, i(sic87dd) j(year)

* Merge in employment weights
merge m:1 sic87dd using "`weights'.dta", assert(3) nogenerate

* For the 2002 and 2007 measures, we take investment measures directly from the CM data
rename nom_compinvest_cm1992 nom_compinvest1992
rename nom_compinvest_cm2002 nom_compinvest2002
rename nom_compinvest_cm2007 nom_compinvest2007

drop nom_invest2002 nom_invest2007
rename nom_invest_cm2002 nom_invest2002
rename nom_invest_cm2007 nom_invest2007

order sic87dd wt comp_broad nom_compinvest* nom_invest* real_vship* nom_vship* emp* pay* prodpay*

* Define our IT variables for each year
foreach y of numlist 1977 1982 1987 1992 2002 2007 {
	gen ci`y' = 100 * (nom_compinvest`y'/nom_invest`y')
	drop nom_compinvest`y' nom_invest`y'
}

* For 1977, 1982, 1987, and 2002 we have no missing values
foreach y of numlist 1977 1982 1987 2002 {
	assert ci`y' < .
}

* For 1992 and 2007, we have some missing values - but never within the computer sector
assert (ci1992 < .) & (ci2007 < .) if comp_broad == 1

* Interpolate missing values for 1992
replace ci1992 = ((2/3) * ci1987) + ((1/3) * ci2002) if ci1992 == .

* Extrapolate missing values for 2007
quietly sum ci2002 if ci2007 < . [aw = wt]
local ci2002_mean = r(mean)

quietly sum ci2007 if ci2007 < . [aw = wt]
local ci2007_mean = r(mean)

replace ci2007 = ci2002 * (`ci2007_mean'/`ci2002_mean') if ci2007 == .

* Verify that computer investment rates always lie between 0 and 100 (and by implication are always populated)
foreach y of numlist 1977 1982 1987 1992 2002 2007 {
	assert ci`y' >= 0 & ci`y' <= 100
}

/* Prepare data on outcomes */

reshape long ci real_vship nom_vship emp pay prodpay, i(sic87dd) j(year)

* Compute labor productivity
gen laborprod = real_vship/emp

* Compute non-production workers' payroll
gen nonprodpay = pay - prodpay
assert nonprodpay >= 0 & nonprodpay < .

reshape wide ci laborprod real_vship nom_vship emp pay prodpay nonprodpay, i(sic87dd) j(year)
order sic87dd comp_broad ci* laborprod* real_vship* nom_vship* emp* pay* prodpay* nonprodpay*

* Compute annual (non-log) changes in computer investment rates
foreach t in "1977_1982" "1982_1987" "1987_1992" "1992_2002" "2002_2007" {
	local start = substr("`t'", 1, 4)
	local end = substr("`t'", 6, 4)

	gen d_ci_`t' = (ci`end' - ci`start') / (`end' - `start')
	assert d_ci_`t' != .
}

* Compute annual log changes in all outcome variables
foreach v in "laborprod" "real_vship" "nom_vship" "emp" "pay" "prodpay" "nonprodpay" {
	foreach t in "1977_1982" "1982_1987" "1987_1992" "1992_2002" "2002_2007" {
		local start = substr("`t'", 1, 4)
		local end = substr("`t'", 6, 4)
	
		gen dlog_`v'_`t' = 100 * (log(`v'`end') - log(`v'`start')) / (`end' - `start')
		assert dlog_`v'_`t' != .
	}
}

keep sic87dd comp_broad wt d_* dlog_*
reshape long d_ci_ dlog_laborprod_ dlog_real_vship_ dlog_nom_vship_ dlog_emp_ dlog_pay_ dlog_prodpay_ dlog_nonprodpay_, i(sic87dd) j(period) string

rename d_ci_ d_ci
foreach v in "laborprod" "real_vship" "nom_vship" "emp" "pay" "prodpay" "nonprodpay" {
	rename dlog_`v'_ dlog_`v'
}

label var d_ci "Chg in Comp Invest Rate (%)"
label var dlog_laborprod "Real Shipments / Worker"
label var dlog_real_vship "Real Shipments"
label var dlog_nom_vship "Nominal Shipments"
label var dlog_emp "Emp"
label var dlog_pay "Real Wage Bill"
label var dlog_prodpay "Prod Real Wage Bill"
label var dlog_nonprodpay "Non-Prod Real Wage Bill"

* Time effects
quietly levels period, local(periods)
foreach t in "1977_1982" "1982_1987" "1987_1992" "1992_2002" "2002_2007" {
	local start = substr("`t'", 1, 4)
	local end = substr("`t'", 6, 4)

	gen t_`t' = (period == "`t'")
	label var t_`t' "1{`start'-`end'}"
}

local time_effects "t_1977_1982 t_1982_1987 t_1987_1992 t_1992_2002 t_2002_2007"

tempfile core
save "`core'.dta", replace

/* Regression specifications pooling across periods */

use "`core'.dta", clear
estimates clear

foreach v in "laborprod" "real_vship" "nom_vship" "emp" "pay" "prodpay" "nonprodpay" {
	foreach nocomp of numlist 0 1 {
		* Show a spec including the computer sector only for labor productivity
		if ("`v'" != "laborprod") & (`nocomp' == 0) {
			continue
		}	
	
		if `nocomp' == 0 {
			local if_cond ""
			local exclude_comp "No"
		}
		else if `nocomp' == 1 {
			local if_cond "if comp_broad == 0"
			local exclude_comp "Yes"
		}
	
		eststo: reg dlog_`v' d_ci `time_effects' `if_cond' [aw = wt], noconstant vce(cluster sic87dd)
		estadd local exclude_comp "`exclude_comp'"
		
		quietly distinct sic87dd `if_cond'
		estadd scalar num_inds = r(ndistinct)
	}
}

tempfile specs

#delimit ;
esttab * using "`specs'.csv",
	depvars
	collabels(, none)
	b(%9.2f) se(%9.2f)
	star(* .10 ** .05 *** .01)
	nonotes
	order(d_ci t_1977_1982 t_1982_1987 t_1987_1992 t_1992_2002 t_2002_2007)
	label
	stats(exclude_comp num_inds N, labels("Exclude Computer Sector" "Number of Industries" "N") fmt(0 0 0))
	replace;
#delimit cr

* Switch the output to Excel format
insheet using "`specs'.csv", comma clear
export excel "../../out/appendix/tabs/Appendix-Changes-on-Changes.xlsx", sheetreplace sheet(Raw)
rm "`specs'.csv"



