/*
	Description: Constructs the figures and tables included in the online appendix (except Table A4)
	
	Author: Brendan Price
	Prepared for public disclosure: 4/28/14

	NB: Adapted from the file "appendix-core-v2.do" in our own file system.
*/

version 12.1
clear all
set more off
set matsize 11000
set trace off
capture restore, not
capture log close
capture postclose coefs

*** Program to determine axis ranges ***
capture program drop set_axis_ranges
program define set_axis_ranges, rclass
	local step = `1'
	local tick = `2'

	local ymin = 0
	local ymax = 0

	local i = 3
	while "``i''" != "" {
		foreach m in "min" "max" {
			if "`m'" == "min" {
				local d "floor"
				local g "<"
			}
			else if "`m'" == "max" {
				local d "ceil"
				local g ">"
			}

			quietly sum ``i''
			local y`m'_``i'' = `d'(r(`m')/`step') * `step'
			if `y`m'_``i''' `g' `y`m'' {
				local y`m' = `y`m'_``i'''
			}
		}
		
		local i = `i' + 1
	}

	return local yscale = "yscale(range(`ymin' `ymax'))"
	return local ylabel =  "ylabel(`ymin'(`step')`ymax')"
	return local ytick =  "ytick(`ymin'(`tick')`ymax')"
end

/* Construct employment weights: industry's average share of manufacturing employment over the sample period */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep sic87dd year emp
keep if year >= 1980 & year <= 2009
bysort year: egen totemp = total(emp)
gen empsh = emp/totemp
bysort sic87dd: egen wt = mean(empsh)
keep sic87dd wt
duplicates drop
bysort sic87dd: assert _N == 1
egen totwt = total(wt)
assert abs(totwt - 1) < .001
keep sic87dd wt
tempfile weights
save "`weights'.dta", replace

/* Define IT intensity as computer investments/total investments */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep if year == 1977 | year == 1982 | year == 1987 | year == 1992
keep sic87dd year comp_broad nom_compinvest* nom_invest nom_invest_cm2002 nom_invest_cm2007 smtshare_1988 smtshare_1993
reshape wide nom_invest, i(sic87dd) j(year)

* Merge in employment weights
merge m:1 sic87dd using "`weights'.dta", assert(3) nogenerate

* Where applicable, take investment measures directly from the CM data
rename nom_compinvest_cm1992 nom_compinvest1992
rename nom_compinvest_cm2002 nom_compinvest2002
rename nom_compinvest_cm2007 nom_compinvest2007
rename nom_invest_cm2002 nom_invest2002
rename nom_invest_cm2007 nom_invest2007
order sic87dd wt comp_broad nom_invest* nom_compinvest* smtshare*

* Define our IT variables for each year
foreach y of numlist 1977 1982 1987 1992 2002 2007 {
	gen ci`y' = 100 * (nom_compinvest`y'/nom_invest`y')
	assert ci`y' >= 0 & ci`y' <= 100 if ci`y' < .
}

keep sic87dd wt comp_broad ci* smtshare*

* For 1977, 1982, 1987, and 2002 we have no missing values
foreach y of numlist 1977 1982 1987 2002 {
	assert ci`y' < .
}

* For 1992 and 2007, we have some missing values - but never within the computer sector
assert (ci1992 < .) & (ci2007 < .) if comp_broad == 1

* Interpolate missing values for 1992
replace ci1992 = ((2/3) * ci1987) + ((1/3) * ci2002) if ci1992 == .

* Extrapolate missing values for 2007
quietly sum ci2002 if ci2007 < . [aw = wt]
local ci2002_mean = r(mean)

quietly sum ci2007 if ci2007 < . [aw = wt]
local ci2007_mean = r(mean)

replace ci2007 = ci2002 * (`ci2007_mean'/`ci2002_mean') if ci2007 == .

* Preferred measure of computer investments: weighted average of 77/82/87/92/02/07, correcting for the absence of 97
assert 4 * (5/32) + 2 * (3/16) == 1
assert ((5/32) * (1977 + 1982 + 1987 + 1992)) + ((3/16) * (2002 + 2007)) == 1992
gen cimean = ((5/32) * (ci1977 + ci1982 + ci1987 + ci1992)) + ((3/16) * (ci2002 + ci2007))

* Also average computer investment rates across pairs of neighboring years
gen ci7782 = (ci1977 + ci1982)/2
gen ci8792 = (ci1987 + ci1992)/2
gen ci0207 = (ci2002 + ci2007)/2

* Verify that computer investment rates always lie between 0 and 100 (and by implication are always populated)
foreach v of varlist ci* {
	assert `v' >= 0 & `v' <= 100
}

* Take the mean of the SMT technology measure across 1988 and 1993
gen smt = (smtshare_1988 + smtshare_1993)/2
drop smtshare_1988 smtshare_1993

keep sic87dd comp_broad wt ci* smt
tempfile ci_measures_raw
save "`ci_measures_raw'.dta", replace

* Standardize each CI measure to have zero weighted mean and unit weighted SD
foreach v of varlist ci* {
	quietly sum `v' [aw = wt]
	replace `v' = (`v' - r(mean))/r(sd)
}

* Standardize the SMT measure to have zero weighted mean and unit weighted SD (among industries for which it is defined)
quietly sum smt if smt < . [aw = wt]
replace smt = (smt - r(mean))/r(sd) if smt < .

tempfile ci_measures
save "`ci_measures'.dta", replace

/* Create a table showing summary statistics on our computer investment measures (prior to standardization) */

use "`ci_measures_raw'.dta", clear

capture mat drop ITstats
foreach v of varlist cimean ci7782 ci8792 ci0207 smt {
	* Compute summary statistics both in the full sample and for non-computer industries
	foreach nocomp of numlist 0 1 {
		if `nocomp' == 0 {
			local if_cond ""
		}
		else if `nocomp' == 1{
			local if_cond "if comp_broad == 0"
		}
	
		quietly sum `v' `if_cond' [aw = wt], detail

		local n`nocomp' = r(N)
		local mean`nocomp' = r(mean)
		local sd`nocomp' = r(sd)
		local median`nocomp' = r(p50)
		local min`nocomp' = r(min)
		local max`nocomp' = r(max)
	}
	
	* Row of means, medians, mins, maxes
	mat row1 = (`n0', `mean0', `median0', `min0', `max0', `n1', `mean1', `median1', `min1', `max1')
	mat rownames row1 = "`v'"

	* Row of standard deviations
	mat row2 = (., `sd0', ., ., ., ., `sd1', ., ., .)
	mat rownames row2 = "`v'"
	
	* Tack on to stats matrix
	capture confirm mat ITstats
	if _rc != 0 {
		mat ITstats = row1
	}
	else {
		mat ITstats = (ITstats \ row1)
	}
	mat ITstats = (ITstats \ row2)
}

* Load a dataset of summary stats into memory
clear
svmat ITstats

* Add a column indicating variable names
local names: rownames ITstats
mat drop ITstats

gen v = ""
order v

forvalues i = 1/`=_N' {
	local name: word `i' of `names'
	replace v = "`name'" in `i'
}

* Export a table of summary statistics
export excel "../../out/appendix/tabs/Appendix-Summary-Stats.xlsx", sheetreplace sheet(Raw-IT-Stats)

/* Create a table showing pairwise correlations between our core computer investment measures */

use "`ci_measures'.dta", clear

* Correlation table (NB: the option "listwise" in makematrix prevents Stata from restricting all correlations to the SMT sample)
pwcorr cimean ci7782 ci8792 ci0207 smt if comp_broad == 0 [aw = wt]
makematrix IT_rhos, from(r(rho)) listwise: corr cimean ci7782 ci8792 ci0207 smt if comp_broad == 0 [aw = wt]
clear
svmat IT_rhos, names(eqcol)

local names : rownames IT_rhos
gen ITvar = ""
forvalues i = 1/`: word count `names'' {
	local v : word `i' of `names'
	replace ITvar = "`v'" in `i'
}

foreach v in "cimean" "ci7782" "ci8792" "ci0207" "smt" {
	rename _`v' `v'
}

order ITvar cimean ci7782 ci8792 ci0207 smt
export excel "../../out/appendix/tabs/Appendix-Summary-Stats.xlsx", sheetreplace sheet(Raw-Correlations)

/* Prepare data on outcomes */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear

* Restrict to data we need
keep if year >= 1980 & year <= 2009
keep sic87dd year comp_broad real_vship nom_vship emp real_pay real_prodpay piship nom_vadd tfp
rename real_pay pay
rename real_prodpay prodpay

* Compute labor productivity
gen laborprod = real_vship/emp

* Deal with the one case in which real production payroll exceeds real total payroll
assert year == 1995 & sic87dd == 3263 if prodpay > pay
replace prodpay = pay if year == 1995 & sic87dd == 3263

gen nonprodpay = pay - prodpay
assert nonprodpay >= 0 & nonprodpay < .

* Compute logs of all outcome variables
keep sic87dd year comp_broad laborprod real_vship nom_vship emp pay prodpay nonprodpay piship nom_vadd tfp

foreach v of varlist laborprod real_vship nom_vship emp pay prodpay nonprodpay piship nom_vadd tfp {
	* Log variables will always be defined, except for one industry in two years
	if "`v'" == "nonprodpay" {
		assert (`v' > 0 & `v' < .) if (sic87dd != 3263) | ((sic87dd == 3263 & (year != 1995 & year != 1996)))
	}
	else {
		assert `v' > 0 & `v' < .
	}

	* Log levels
	gen log_`v' = 100 * log(`v')
}

label var log_laborprod "Real Shipments/Worker"
label var log_real_vship "Real Shipments"
label var log_nom_vship "Nominal Shipments"
label var log_emp "Employment"
label var log_pay "Wage Bill"
label var log_prodpay "Prod Wage Bill"
label var log_nonprodpay "Non-Prod Wage Bill"
label var log_piship "Shipments Deflator"
label var log_nom_vadd "Nominal Value Added"
label var log_tfp "TFP"

keep sic87dd year comp_broad log_*

* Merge in employment weights and measures of IT intensity
merge m:1 sic87dd using "`ci_measures'.dta", assert(3) nogenerate

* Industry dummies
quietly tab sic87dd, gen(ind_fe)

* Year dummies (omitting the base year)
forvalues y = 1981/2009 {
	gen time_`y' = (year == `y')
}

order sic87dd year wt ci* smt log_* time_* ind_fe* 

tempfile core
save "`core'.dta", replace

/* Create a table of summary statistics for our outcomes */

capture mat drop outcomestats

use "`core'.dta", clear

keep sic87dd year comp_broad wt log_laborprod log_real_vship log_nom_vship log_emp log_pay
keep if year == 1980 | year == 1990 | year == 2000 | year == 2009

* Express as actual logs, not log points
foreach v of varlist log_* {
	replace `v' = `v'/100
}
	
foreach v of varlist log_laborprod log_real_vship log_nom_vship log_emp log_pay {
	foreach limit_to_sic3438 of numlist 0 1 {
		if `limit_to_sic3438' == 0 {
			local sic_restriction "1"
		}
		else if `limit_to_sic3438' == 1 {
			local sic_restriction "sic87dd >= 3400 & sic87dd <= 3899"
		}
	
		foreach y of numlist 1980 1990 2000 2009 {
			* For labor productivity, compute results including and excluding the computer sector
			if "`v'" == "log_laborprod" {
				quietly sum `v' if (year == `y') & (`sic_restriction') [aw = wt]
			
				local mean`limit_to_sic3438'_allinds_`y' = r(mean)
				local sd`limit_to_sic3438'_allinds_`y' = r(sd)
			}
		
			* For other outcomes, compute results only excluding the computer sector
			quietly sum `v' if (year == `y') & (`sic_restriction') & (comp_broad == 0) [aw = wt]

			local mean`limit_to_sic3438'_nocomp_`y' = r(mean)
			local sd`limit_to_sic3438'_nocomp_`y' = r(sd)
		}
	
		* Sample size with and without computer sector
		quietly distinct sic87dd if `sic_restriction'
		local n`limit_to_sic3438'_allinds = r(ndistinct)
		quietly distinct sic87dd if (`sic_restriction') & (comp_broad == 0)
		local n`limit_to_sic3438'_nocomp = r(ndistinct)
	}
	
	* Row of means, medians, mins, maxes
	foreach c in "allinds" "nocomp" {
		if ("`v'" != "log_laborprod") & ("`c'" == "allinds") {
			continue
		}
	
		mat row1 = (`n0_`c'', `mean0_`c'_1980', `mean0_`c'_1990', `mean0_`c'_2000', `mean0_`c'_2009', `n1_`c'', `mean1_`c'_1980', `mean1_`c'_1990', `mean1_`c'_2000', `mean1_`c'_2009')
		mat rownames row1 = "`v'_`c'"

		* Row of standard deviations
		mat row2 = (., `sd0_`c'_1980', `sd0_`c'_1990', `sd0_`c'_2000', `sd0_`c'_2009', ., `sd1_`c'_1980', `sd1_`c'_1990', `sd1_`c'_2000', `sd1_`c'_2009')
		mat rownames row2 = "`v'_`c'"

		* Tack on to stats matrix
		capture confirm mat outcomestats
		if _rc != 0 {
			mat outcomestats = row1
		}
		else {
			mat outcomestats = (outcomestats \ row1)
		}
		mat outcomestats = (outcomestats \ row2)
	}
}

* Load a dataset of summary stats into memory
clear
svmat outcomestats

* Add a column indicating variable names
local names: rownames outcomestats
mat drop outcomestats

gen v = ""
order v

forvalues i = 1/`=_N' {
	local name: word `i' of `names'
	replace v = "`name'" in `i'
}

* Export a table of summary statistics
export excel "../../out/appendix/tabs/Appendix-Summary-Stats.xlsx", sheetreplace sheet(Raw-Outcome-Stats)

/* Estimate the association between various IT measures and various outcomes of interest */

foreach ITvar in "cimean" "ci7782" "ci8792" "ci0207" "smt" {
	* Specify which industries are to be included in the sample
	if "`ITvar'" == "smt" {
		local in_sample "sic87dd >= 3400 & sic87dd <= 3899"
	}
	else {
		local in_sample "1"
	}

	use "`core'.dta", clear

	* Exclude the computer sector: all of the appendix figures are based on results that exclude this sector
	drop if comp_broad == 1

	* Year x IT interactions
	forvalues i = 1981/2009 {
		gen IT_`i' = `ITvar' * time_`i'
	}
	
	keep sic87dd year wt log_* IT_* time_* ind_fe* 
	order sic87dd year wt log_* IT_* time_* ind_fe* 

	foreach d in "laborprod" "real_vship" "nom_vship" "emp" "piship" "nom_vadd" "tfp" "prodpay" "nonprodpay" {
		if "`d'" == "laborprod" | "`d'" == "real_vship" | "`d'" == "nom_vship" | "`d'" == "emp" {
			if "`ITvar'" == "smt" continue
		}
		else if "`d'" == "piship" | "`d'" == "nom_vadd" | "`d'" == "tfp" | "`d'" == "prodpay" | "`d'" == "nonprodpay" {
			if "`ITvar'" == "ci7782" | "`ITvar'" == "ci8792" | "`ITvar'" == "ci0207" continue
		}

		postfile coefs str6 ITvar str10 depvar int year double beta double se using "../../out/appendix/dta/estimates-`ITvar'-`d'.dta", replace

		* Run the spec
		reg log_`d' IT_* ind_fe* time_* if `in_sample' [aw = wt], noconstant vce(cluster sic87dd)

		forvalues i = 1980/2009 {
			* Normalize (fictitious) first coefficient to zero
			if `i' == 1980 {
				post coefs ("`ITvar'") ("`d'") (`i') (0) (0)
			}
			else {
				local beta = _b[IT_`i']
				local se = _se[IT_`i']

				post coefs ("`ITvar'") ("`d'") (`i') (`beta') (`se')
			}
		}

		postclose coefs
	}
}

* Combine estimates from different specifications
foreach ITvar in "cimean" "ci7782" "ci8792" "ci0207" "smt" {
	foreach d in "laborprod" "real_vship" "nom_vship" "emp" "piship" "nom_vadd" "tfp" "prodpay" "nonprodpay" {
		if "`ITvar'" == "cimean" & "`d'" == "laborprod" {
			use "../../out/appendix/dta/estimates-`ITvar'-`d'.dta", clear
		}
		else {
			capture append using "../../out/appendix/dta/estimates-`ITvar'-`d'.dta"
		}
	}
}

* Confidence intervals
gen beta_low = beta - (1.96 * se)
gen beta_high = beta + (1.96 * se)

save "../../out/appendix/dta/estimates.dta", replace

/* For our core outcomes, plot results based on different vintages of our computer investment measure */

foreach d in "laborprod" "real_vship" "nom_vship" "emp" {
	if "`d'" == "laborprod" {
		local d_lbl "LaborProd"
	}
	else if "`d'" == "real_vship" {
		local d_lbl "RealShip"
	}
	else if "`d'" == "nom_vship" {
		local d_lbl "NomShip"
	}
	else if "`d'" == "emp" {
		local d_lbl "Emp"
	}

	* Extract coefficients using four different computer investment measures
	use "../../out/appendix/dta/estimates.dta", clear
	keep if (depvar == "`d'") & (ITvar != "smt")
	keep ITvar year beta
	rename beta beta_
	reshape wide beta_, i(year) j(ITvar) string

	* Specify axis limits
	if "`d'" == "laborprod" {
		local step 2
		local tick 1
	}
	else {
		local step 5
		local tick 2.5
	}

	set_axis_ranges	`step' `tick' beta_cimean beta_ci7782 beta_ci8792 beta_ci0207
	local yscale "`=r(yscale)'"
	local ylabel "`=r(ylabel)'"
	local ytick "`=r(ytick)'"

	#delimit ;
	twoway
		(connected beta_cimean year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
		(connected beta_ci7782 year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(longdash))
		(connected beta_ci8792 year, mcolor(dkgreen) msymbol(square) msize(medsmall) clcolor(dkgreen) clpattern(dash))
		(connected beta_ci0207 year, mcolor(dkorange) msymbol(plus) msize(medsmall) clcolor(dkorange) clpattern(shortdash)),
		xtitle("")
		ytitle("Coef. on IT Measure x Year Dummy", size(medium))
		xscale(range(1980 2009))
		xlabel(1980(5)2005)
		xtick(1980(1)2009)
		`yscale'
		`ylabel'
		`ytick'
		graphregion(color(white))
		legend(rows(2) label(1 "1977-2007 Comp Investments") label(2 "1977/1982 Comp Investments") label(3 "1987/1992 Comp Investments") label(4 "2002/2007 Comp Investments") size(small));
	#delimit cr

	graph export "../../out/appendix/figs/App-Figure-CompInvest-Vintage-`d_lbl'.eps", as(eps) replace
	!epstopdf --outfile="../../out/appendix/figs/App-Figure-CompInvest-Vintage-`d_lbl'.pdf" "../../out/appendix/figs/App-Figure-CompInvest-Vintage-`d_lbl'.eps"
}

/* For additional outcomes of interest, plot results based on our core computer investment and SMT measures */

foreach d in "nom_vadd" "piship" "tfp" {
	if "`d'" == "nom_vadd" {
		local d_lbl "NomVA"
	}
	else if "`d'" == "piship" {
		local d_lbl "Prices"
	}
	else if "`d'" == "tfp" {
		local d_lbl "TFP"
	}

	* Extract coefficients using four different computer investment measures
	use "../../out/appendix/dta/estimates.dta", clear
	keep if (depvar == "`d'") & ((ITvar == "cimean") | (ITvar == "smt"))
	keep ITvar year beta beta_low beta_high
	rename beta beta_
	rename beta_low beta_low_
	rename beta_high beta_high_
	reshape wide beta_ beta_low_ beta_high_, i(year) j(ITvar) string

	* Specify axis limits
	if "`d'" == "nom_vadd" {
		local step 10
		local tick 5
	}
	else {
		local step 5
		local tick 2.5
	}
	
	set_axis_ranges	`step' `tick' beta_cimean beta_low_cimean beta_high_cimean beta_smt beta_low_smt beta_high_smt
	local yscale "`=r(yscale)'"
	local ylabel "`=r(ylabel)'"
	local ytick "`=r(ytick)'"

	#delimit ;
	twoway
		(connected beta_cimean year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
		(line beta_low_cimean year, clcolor(navy) clpattern(shortdash))
		(line beta_high_cimean year, clcolor(navy) clpattern(shortdash))
		(connected beta_smt year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
		(line beta_low_smt year, clcolor(maroon) clpattern(longdash))
		(line beta_high_smt year, clcolor(maroon) clpattern(longdash)),	
		xtitle("")
		ytitle("Coef. on IT Measure x Year Dummy", size(medium))
		xscale(range(1980 2009))
		xlabel(1980(5)2005)
		xtick(1980(1)2009)
		`yscale'
		`ylabel'
		`ytick'
		graphregion(color(white))
		legend(rows(2) order(1 2 4 5) label(1 "1977-2007 Comp Investments") label(2 "95 Percent CI") label(4 "1988/1993 Technology Usage") label(5 "95 Percent CI") size(small));
	#delimit cr

	graph export "../../out/appendix/figs/App-Figure-CompInvest-vs-SMT-`d_lbl'.eps", as(eps) replace
	!epstopdf --outfile="../../out/appendix/figs/App-Figure-CompInvest-vs-SMT-`d_lbl'.pdf" "../../out/appendix/figs/App-Figure-CompInvest-vs-SMT-`d_lbl'.eps"
}

/* For production/non-production workers, plot both outcomes together using both computer investments and SMT */

foreach ITvar in "cimean" "smt" {
	if "`ITvar'" == "cimean" {
		local ITvar_lbl "CompInvest"
	}
	else if "`ITvar'" == "smt" {
		local ITvar_lbl "SMT"
	}

	* Extract coefficients using four different computer investment measures
	use "../../out/appendix/dta/estimates.dta", clear
	keep if (depvar == "prodpay" | depvar == "nonprodpay") & (ITvar == "`ITvar'")
	keep depvar year beta beta_low beta_high
	rename beta beta_
	rename beta_low beta_low_
	rename beta_high beta_high_
	reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

	* Specify axis limits
	set_axis_ranges	10 5 beta_nonprodpay beta_low_nonprodpay beta_high_nonprodpay beta_prodpay beta_low_prodpay beta_high_prodpay
	local yscale "`=r(yscale)'"
	local ylabel "`=r(ylabel)'"
	local ytick "`=r(ytick)'"

	#delimit ;
	twoway
		(connected beta_nonprodpay year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
		(line beta_low_nonprodpay year, clcolor(navy) clpattern(shortdash))
		(line beta_high_nonprodpay year, clcolor(navy) clpattern(shortdash))
		(connected beta_prodpay year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
		(line beta_low_prodpay year, clcolor(maroon) clpattern(longdash))
		(line beta_high_prodpay year, clcolor(maroon) clpattern(longdash)),	
		xtitle("")
		ytitle("Coef. on IT Measure x Year Dummy", size(medium))
		xscale(range(1980 2009))
		xlabel(1980(5)2005)
		xtick(1980(1)2009)
		`yscale'
		`ylabel'
		`ytick'
		graphregion(color(white))
		legend(rows(2) order(1 2 4 5) label(1 "Non-Production Wage Bill") label(2 "95 Percent CI") label(4 "Production Wage Bill") label(5 "95 Percent CI") size(small));
	#delimit cr

	graph export "../../out/appendix/figs/App-Figure-`ITvar_lbl'-NonProd-Prod.eps", as(eps) replace
	!epstopdf --outfile="../../out/appendix/figs/App-Figure-`ITvar_lbl'-NonProd-Prod.pdf" "../../out/appendix/figs/App-Figure-`ITvar_lbl'-NonProd-Prod.eps"
}

/* Unreported analysis: study the relationship between computer investments and non-production/production labor in the restricted set of SIC 34-38 industries */

use "`core'.dta", clear
drop if comp_broad == 1
keep if sic87dd >= 3400 & sic87dd <= 3899 

* Year x IT interactions
forvalues i = 1981/2009 {
	gen IT_`i' = cimean * time_`i'
}
	
keep sic87dd year wt log_* IT_* time_* ind_fe* 
order sic87dd year wt log_* IT_* time_* ind_fe* 

* Run the spec
foreach d in "prodpay" "nonprodpay" {
	postfile coefs str6 ITvar str10 depvar int year double beta double se using "../../out/appendix/dta/estimates-cimean-`d'-sic3438.dta", replace

	* Run the spec
	reg log_`d' IT_* ind_fe* time_* [aw = wt], noconstant vce(cluster sic87dd)

	forvalues i = 1980/2009 {
		* Normalize (fictitious) first coefficient to zero
		if `i' == 1980 {
			post coefs ("cimean") ("`d'") (`i') (0) (0)
		}
		else {
			local beta = _b[IT_`i']
			local se = _se[IT_`i']

			post coefs ("cimean") ("`d'") (`i') (`beta') (`se')
		}
	}

	postclose coefs
}

use "../../out/appendix/dta/estimates-cimean-prodpay-sic3438.dta", clear
append using "../../out/appendix/dta/estimates-cimean-nonprodpay-sic3438.dta"

gen beta_low = beta - (1.96 * se)
gen beta_high = beta + (1.96 * se)

keep depvar year beta beta_low beta_high
rename beta beta_
rename beta_low beta_low_
rename beta_high beta_high_
reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

* Specify axis limits
set_axis_ranges	10 5 beta_nonprodpay beta_low_nonprodpay beta_high_nonprodpay beta_prodpay beta_low_prodpay beta_high_prodpay
local yscale "`=r(yscale)'"
local ylabel "`=r(ylabel)'"
local ytick "`=r(ytick)'"

#delimit ;
twoway
	(connected beta_nonprodpay year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low_nonprodpay year, clcolor(navy) clpattern(shortdash))
	(line beta_high_nonprodpay year, clcolor(navy) clpattern(shortdash))
	(connected beta_prodpay year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low_prodpay year, clcolor(maroon) clpattern(longdash))
	(line beta_high_prodpay year, clcolor(maroon) clpattern(longdash)),	
	title("1977-2007 Computer Investments and Non-Production/Production Labor", size(medsmall))
	subtitle("Sample Restricted to SIC 34-38", size(medsmall))
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xscale(range(1980 2009))
	xlabel(1980(5)2005)
	xtick(1980(1)2009)
	`yscale'
	`ylabel'
	`ytick'
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "Non-Production Wage Bill") label(2 "95 Percent CI") label(4 "Production Wage Bill") label(5 "95 Percent CI") size(small));
#delimit cr

graph export "../../out/appendix/figs/App-Figure-CompInvest-NonProd-Prod-SIC3438.eps", as(eps) replace
!epstopdf --outfile="../../out/appendix/figs/App-Figure-CompInvest-NonProd-Prod-SIC3438.pdf" "../../out/appendix/figs/App-Figure-CompInvest-NonProd-Prod-SIC3438.eps"



