/*
	Description: Constructs the figures included in the main body of the paper

	Author: Brendan Price
	Prepared for public disclosure: 4/27/14
	
	NB: Adapted from the file "draft-v5.do" in our own file system.
*/

version 12.1
clear all
set more off
set matsize 11000
set trace off
capture restore, not
capture log close
capture postclose coefs

/* Construct employment weights: industry's average share of manufacturing employment over the sample period */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep sic87dd year emp
keep if year >= 1980 & year <= 2009
bysort year: egen totemp = total(emp)
gen empsh = emp/totemp
bysort sic87dd: egen wt = mean(empsh)
keep sic87dd wt
duplicates drop
bysort sic87dd: assert _N == 1
egen totwt = total(wt)
assert abs(totwt - 1) < .001
keep sic87dd wt
tempfile weights
save "`weights'.dta", replace

/* Define IT intensity as computer investments/total investments */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear
keep if year == 1977 | year == 1982 | year == 1987 | year == 1992
keep sic87dd year comp_broad nom_compinvest* nom_invest nom_invest_cm2002 nom_invest_cm2007 smtshare_1988 smtshare_1993
reshape wide nom_invest, i(sic87dd) j(year)

* Merge in employment weights
merge m:1 sic87dd using "`weights'.dta", assert(3) nogenerate

* Where applicable, take investment measures directly from the CM data
rename nom_compinvest_cm1992 nom_compinvest1992
rename nom_compinvest_cm2002 nom_compinvest2002
rename nom_compinvest_cm2007 nom_compinvest2007
rename nom_invest_cm2002 nom_invest2002
rename nom_invest_cm2007 nom_invest2007
order sic87dd wt comp_broad nom_invest* nom_compinvest* smtshare*

* Define our IT variables for each year
foreach y of numlist 1977 1982 1987 1992 2002 2007 {
	gen ci`y' = 100 * (nom_compinvest`y'/nom_invest`y')
	assert ci`y' >= 0 & ci`y' <= 100 if ci`y' < .
}

keep sic87dd wt comp_broad ci* smtshare*

* For 1977, 1982, 1987, and 2002 we have no missing values
foreach y of numlist 1977 1982 1987 2002 {
	assert ci`y' < .
}

* For 1992 and 2007, we have some missing values - but never within the computer sector
assert (ci1992 < .) & (ci2007 < .) if comp_broad == 1

* Interpolate missing values for 1992
replace ci1992 = ((2/3) * ci1987) + ((1/3) * ci2002) if ci1992 == .

* Extrapolate missing values for 2007
quietly sum ci2002 if ci2007 < . [aw = wt]
local ci2002_mean = r(mean)

quietly sum ci2007 if ci2007 < . [aw = wt]
local ci2007_mean = r(mean)

replace ci2007 = ci2002 * (`ci2007_mean'/`ci2002_mean') if ci2007 == .

* Preferred measure of computer investments: weighted average of 77/82/87/92/02/07, correcting for the absence of 97
assert 4 * (5/32) + 2 * (3/16) == 1
assert ((5/32) * (1977 + 1982 + 1987 + 1992)) + ((3/16) * (2002 + 2007)) == 1992
gen cimean = ((5/32) * (ci1977 + ci1982 + ci1987 + ci1992)) + ((3/16) * (ci2002 + ci2007))

* Also average computer investment rates across pairs of neighboring years
gen ci7782 = (ci1977 + ci1982)/2
gen ci8792 = (ci1987 + ci1992)/2
gen ci0207 = (ci2002 + ci2007)/2

* Verify that computer investment rates always lie between 0 and 100 (and by implication are always populated)
foreach v of varlist ci* {
	assert `v' >= 0 & `v' <= 100
}

* Take the mean of the SMT technology measure across 1988 and 1993
gen smt = (smtshare_1988 + smtshare_1993)/2
drop smtshare_1988 smtshare_1993

* Standardize each CI measure to have zero weighted mean and unit weighted SD
foreach v of varlist ci* {
	quietly sum `v' [aw = wt]
	replace `v' = (`v' - r(mean))/r(sd)
}

* Standardize the SMT measure to have zero weighted mean and unit weighted SD (among industries for which it is defined)
quietly sum smt if smt < . [aw = wt]
replace smt = (smt - r(mean))/r(sd) if smt < .

keep sic87dd wt ci* smt

tempfile ci_measures
save "`ci_measures'.dta", replace

/* Prepare data on outcomes */

use "../../dta/nber-ces/clean/nber-ces-clean.dta", clear

* Restrict to data we need
keep if year >= 1980 & year <= 2009
keep sic87dd year comp_broad real_vship nom_vship emp real_pay
rename real_pay pay

* Compute labor productivity
gen laborprod = real_vship/emp

* Compute logs of all outcome variables
keep sic87dd year comp_broad laborprod real_vship nom_vship emp pay

foreach v of varlist laborprod real_vship nom_vship emp pay {	
	* Log levels
	assert `v' > 0 & `v' < .
	gen log_`v' = 100 * log(`v')
}

label var log_laborprod "Real Shipments/Worker"
label var log_real_vship "Real Shipments"
label var log_nom_vship "Nominal Shipments"
label var log_emp "Employment"
label var log_pay "Wage Bill"

keep sic87dd year comp_broad log_*

* Merge in employment weights and measures of IT intensity
merge m:1 sic87dd using "`ci_measures'.dta", assert(3) nogenerate

* Industry dummies
quietly tab sic87dd, gen(ind_fe)

* Year dummies (omitting the base year)
forvalues y = 1981/2009 {
	gen time_`y' = (year == `y')
}

order sic87dd year wt comp_broad ci* smt log_* time_* ind_fe* 

tempfile core
save "`core'.dta", replace

/* Estimate the association between variuos IT measures and various outcomes of interest */

foreach ITvar in "cimean" "smt" "ci8792" {
	use "`core'.dta", clear

	* Year x IT interactions
	forvalues i = 1981/2009 {
		gen IT_`i' = `ITvar' * time_`i'
	}
	
	keep sic87dd year comp_broad wt log_* IT_* time_* ind_fe* 
	order sic87dd year comp_broad wt log_* IT_* time_* ind_fe* 

	foreach d in "laborprod" "real_vship" "nom_vship" "emp" "pay" {
		* For the 87/92 CI measure, we only run results on labor productivity
		if ("`d'" != "laborprod") & ("`ITvar'" != "cimean") & ("`ITvar'" != "smt") {
			continue
		}

		postfile coefs str6 ITvar str10 depvar int drop_comp int restrict_to_sic_3438 int year double beta double se using "../../out/main-text/dta/estimates-`ITvar'-`d'.dta", replace

		* Loop over whether to exclude the computer sector
		foreach drop_comp of numlist 0 1 {
			* Only run results including the computer sector when we use the mean CI measure
			if (`drop_comp' == 0) & ("`ITvar'" != "cimean") {
				continue
			}

			if `drop_comp' == 0 {
				local comp_cond "1"
			}
			else if `drop_comp' == 1 {
				local comp_cond "comp_broad == 0"
			}

			* Loop over whether to restrict to SICs 34-38
			foreach restrict_to_sic_3438 of numlist 0 1 {
				if ("`ITvar'" == "smt") & (`restrict_to_sic_3438' == 0) {
					continue
				}
				
				* Specify which sectors are to be included
				if `restrict_to_sic_3438' == 0 {
					local sic3438_cond "1"
				}
				else if `restrict_to_sic_3438' == 1 {
					local sic3438_cond "sic87dd >= 3400 & sic87dd <= 3899"
				}

				* Run the spec
				reg log_`d' IT_* ind_fe* time_* if (`comp_cond') & (`sic3438_cond') [aw = wt], noconstant vce(cluster sic87dd)
				
				forvalues i = 1980/2009 {
					* Normalize (fictitious) first coefficient to zero
					if `i' == 1980 {
						post coefs ("`ITvar'") ("`d'") (`drop_comp') (`restrict_to_sic_3438') (`i') (0) (0)
					}
					else {
						local beta = _b[IT_`i']
						local se = _se[IT_`i']

						post coefs ("`ITvar'") ("`d'") (`drop_comp') (`restrict_to_sic_3438') (`i') (`beta') (`se')
					}
				}
			}
		}

		postclose coefs
	}
}

/* Construct the figures that will appear in our paper */

* Combine estimates from different specifications
foreach d in "laborprod" "real_vship" "nom_vship" "emp" "pay" {
	foreach ITvar in "cimean" "smt" "ci8792" {
		if "`d'" == "laborprod" & "`ITvar'" == "cimean" {
			use "../../out/main-text/dta/estimates-`ITvar'-`d'.dta", clear
		}
		else {
			capture append using "../../out/main-text/dta/estimates-`ITvar'-`d'.dta"
		}
	}
}

* Confidence intervals
gen beta_low = beta - (1.96 * se)
gen beta_high = beta + (1.96 * se)

save "../../out/main-text/dta/estimates.dta", replace

/* Figure 1A: Real shipments/worker with and without computer sector, IT = 1977-2007 computer investments */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "laborprod") & (ITvar == "cimean") & (restrict_to_sic_3438 == 0)
reshape wide beta beta_low beta_high se, i(depvar ITvar year) j(drop_comp)

#delimit ;
twoway
	(connected beta0 year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low0 year, clcolor(navy) clpattern(shortdash))
	(line beta_high0 year, clcolor(navy) clpattern(shortdash))
	(connected beta1 year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low1 year, clcolor(maroon) clpattern(longdash))
	(line beta_high1 year, clcolor(maroon) clpattern(longdash)),
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-10 70))
	ylabel(-10(10)70)
	ytick(-10(5)70)
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "All Mfg Industries") label(2 "95 Percent CI") label(4 "Excluding Computer-Producing Industries") label(5 "95 Percent CI") size(medsmall));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-1A.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-1A.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-1A.eps"

/* Figure 1B: Real shipments/worker using 1977-2007 computer investments, 1987/1992 computer investments, and the 1988/1993 SMT measure */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "laborprod") & (ITvar == "cimean" | ITvar == "ci8792" | ITvar == "smt") & (drop_comp == 1) & (restrict_to_sic_3438 == 1)

keep ITvar year beta
rename beta beta_
reshape wide beta_, i(year) j(ITvar) string

#delimit ;
twoway
	(connected beta_cimean year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(connected beta_ci8792 year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(longdash))
	(connected beta_smt year, mcolor(dkgreen) msymbol(square) msize(medsmall) clcolor(dkgreen) clpattern(shortdash)),
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-10 15))
	ylabel(-10(5)15)
	ytick(-10(2.5)15)
	graphregion(color(white))
	legend(rows(3) label(1 "1977-2007 Comp Investments") label(2 "1987/1992 Comp Investments") label(3 "1988/1993 Technology Usage") size(medsmall));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-1B.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-1B.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-1B.eps"

/* Figure 2A: Log real and nominal shipments without computer sector, IT = 1977-2007 computer investments */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "real_vship" | depvar == "nom_vship") & (ITvar == "cimean") & (drop_comp == 1) & (restrict_to_sic_3438 == 0)

keep depvar year beta beta_low beta_high
rename beta beta_
rename beta_low beta_low_
rename beta_high beta_high_
reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

#delimit ;
twoway
	(connected beta_real_vship year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low_real_vship year, clcolor(navy) clpattern(shortdash))
	(line beta_high_real_vship year, clcolor(navy) clpattern(shortdash))
	(connected beta_nom_vship year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low_nom_vship year, clcolor(maroon) clpattern(longdash))
	(line beta_high_nom_vship year, clcolor(maroon) clpattern(longdash)),	
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-35 15))
	ylabel(-35(5)15)
	ytick(-35(2.5)15)
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "Real Shipments") label(2 "95 Percent CI") label(4 "Nominal Shipments") size(medsmall) label(5 "95 Percent CI"));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-2A.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-2A.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-2A.eps"

/* Figure 2B: Log real and nominal shipments without computer sector, using SMT measure */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "real_vship" | depvar == "nom_vship") & (ITvar == "smt") & (drop_comp == 1) & (restrict_to_sic_3438 == 1)

keep depvar year beta beta_low beta_high
rename beta beta_
rename beta_low beta_low_
rename beta_high beta_high_
reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

#delimit ;
twoway
	(connected beta_real_vship year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low_real_vship year, clcolor(navy) clpattern(shortdash))
	(line beta_high_real_vship year, clcolor(navy) clpattern(shortdash))
	(connected beta_nom_vship year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low_nom_vship year, clcolor(maroon) clpattern(longdash))
	(line beta_high_nom_vship year, clcolor(maroon) clpattern(longdash)),
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-35 15))
	ylabel(-35(5)15)
	ytick(-35(2.5)15)
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "Real Shipments") label(2 "95 Percent CI") label(4 "Nominal Shipments") label(5 "95 Percent CI") size(medsmall));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-2B.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-2B.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-2B.eps"

/* Figure 3A: Log employment and log real wage bill, IT = 1977-2007 computer investments */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "emp" | depvar == "pay") & (ITvar == "cimean") & (drop_comp == 1) & (restrict_to_sic_3438 == 0)
keep depvar year beta beta_low beta_high
rename beta beta_
rename beta_low beta_low_
rename beta_high beta_high_
reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

#delimit ;
twoway
	(connected beta_emp year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low_emp year, clcolor(navy) clpattern(shortdash))
	(line beta_high_emp year, clcolor(navy) clpattern(shortdash))
	(connected beta_pay year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low_pay year, clcolor(maroon) clpattern(longdash))
	(line beta_high_pay year, clcolor(maroon) clpattern(longdash)),
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-35 15))
	ylabel(-35(5)15)
	ytick(-35(2.5)15)
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "Employment") label(2 "95 Percent CI") label(4 "Real Wage Bill") label(5 "95 Percent CI") size(medsmall));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-3A.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-3A.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-3A.eps"

/* Figure 3B: Log employment and log real wage bill, using SMT measure */

use "../../out/main-text/dta/estimates.dta", clear
keep if (depvar == "emp" | depvar == "pay") & (ITvar == "smt") & (drop_comp == 1) & (restrict_to_sic_3438 == 1)
keep depvar year beta beta_low beta_high
rename beta beta_
rename beta_low beta_low_
rename beta_high beta_high_
reshape wide beta_ beta_low_ beta_high_, i(year) j(depvar) string

#delimit ;
twoway
	(connected beta_emp year, mcolor(navy) msymbol(circle) msize(medsmall) clcolor(navy) clpattern(solid))
	(line beta_low_emp year, clcolor(navy) clpattern(shortdash))
	(line beta_high_emp year, clcolor(navy) clpattern(shortdash))
	(connected beta_pay year, mcolor(maroon) msymbol(triangle) msize(medsmall) clcolor(maroon) clpattern(solid))
	(line beta_low_pay year, clcolor(maroon) clpattern(longdash))
	(line beta_high_pay year, clcolor(maroon) clpattern(longdash)),
	xtitle("")
	ytitle("Coef. on IT Measure x Year Dummy", size(medium))
	xlabel(1980(4)2009)
	xtick(1980(2)2009)
	yscale(range(-35 15))
	ylabel(-35(5)15)
	ytick(-35(2.5)15)
	graphregion(color(white))
	legend(rows(2) order(1 2 4 5) label(1 "Employment") label(2 "95 Percent CI") label(4 "Real Wage Bill") label(5 "95 Percent CI") size(medsmall));
#delimit cr

graph export "../../out/main-text/figs/Autor-AER-PP-2014-Figure-3B.eps", as(eps) replace
!epstopdf --outfile="../../out/main-text/figs/Autor-AER-PP-2014-Figure-3B.pdf" "../../out/main-text/figs/Autor-AER-PP-2014-Figure-3B.eps"




