*     ************************************************************		*
*	Project: Jones-Olken Project, Climate Shocks and Exports
*	Purpose: Reproduce Table 1 and Table 2 in paper
* Date:    23 December 2009
* Stata:   Version 11.0 MP 
*     ************************************************************		*

clear
clear matrix
set more off
set mem 8G
set mat 11000
set maxvar 12000

capture log close
log using 091223climatetrade, replace


*********************************
*********************************
* Table 1
*********************************
*********************************


******************************
* Loop code over data sets

foreach var of newlist usimport_1D usimport_2d world_export_1D world_export_2D {
	
	use JO_`var'.dta, clear

	* Create prodcut x year indicator variables
	quietly tab prdyr_id, gen(PRDYR)
		
	************************************************************
	* Analysis
	* Iterated FGLS
	* Estimating equation (1) in paper
	
	* Given large number of variables, demean everything first by product x country indicator
	* Demean main regressors
	foreach var2 of varlist g_imp wtem wtem_poor wpre wpre_poor {
		bys prdcnt_id: egen m`var2' = mean(`var2')
		g d`var2' = `var2' - m`var2'
	}
	* Demean product x year regressors
	egen prdyrfe = nvals(prdyr_id)
	sum prdyrfe
	if `r(max)' < 1600 {
		local LASTNUM = `r(max)'
		foreach num of numlist 1/`LASTNUM' {
			disp "`num' of `LASTNUM'"
			bys prdcnt_id: egen double mPRDYR`num' = mean(PRDYR`num')
			g double dPRDYR`num' = PRDYR`num' - mPRDYR`num'
		}
	}
	sum prdyrfe
	if (`r(max)' > 1600) & (`r(max)' < 3200) {
		local LASTNUM = `r(max)'
		foreach num of numlist 1/1600 {
			disp "`num' of `LASTNUM'"
			bys prdcnt_id: egen double mPRDYR`num' = mean(PRDYR`num')
			g double dPRDYR`num' = PRDYR`num' - mPRDYR`num'
		}
		foreach num of numlist 1601/`LASTNUM' {
			disp "`num' of `LASTNUM'"
			bys prdcnt_id: egen mPRDYR`num' = mean(PRDYR`num')
			g dPRDYR`num' = PRDYR`num' - mPRDYR`num'
		}		
	}

	*** Now estimate equation (1)
	* Iteration 1: generate residuals
	reg dg_imp dwtem dwtem_poor dwpre dwpre_poor dPRDYR*
	predict double ehat, resid
	bys prdcnt_id: egen double tmpv = sd(ehat)
	g double invvarehat = tmpv^(-2)
	sum invvarehat, detail
	drop tmpv
		
	* Iteration 2: FGLS round 1
	reg dg_imp dwtem dwtem_poor dwpre dwpre_poor dPRDYR* [w=invvarehat], cluster(cc_num)
	predict double ehat2, resid
	bys prdcnt_id: egen double tmpv = sd(ehat2)
	g double invvarehat2 = tmpv^(-2)
	sum invvarehat2, detail
	drop tmpv
	
	* Iteration 3: FGLS round 2
	reg dg_imp dwtem dwtem_poor dwpre dwpre_poor dPRDYR* [w=invvarehat2], cluster(cc_num)
	lincom dwtem + dwtem_poor
	local tstat=r(estimate) / r(se)
	outreg2 dwtem dwtem_poor dwpre dwpre_poor using results_`var', addstat(poor effect, `r(estimate)', standard error, `r(se)', t-stat, `tstat') replace word

}	

* End of loop over datasets
*********************************






*********************************
*********************************
* Table 2
*********************************
*********************************

use JO_usimport_2D.dta, clear

* Get year fe for regression
tab year, gen(yr)

* Generate observation id number
gen n_obs=_n

* Create sitc id variable to eliminate gaps in the min(sitc)-max(sitc) sequence
sort sitc
tostring sitc, gen(sitc2)
encode sitc2, gen(sitcid)
drop sitc2
sum sitcid
local NN=r(max)
g t_obs = _N
disp "Number of sitc codes is `NN'"

* Variables to save results
gen t2beta=. 
gen t2se=.
gen t2tstat=.
gen t2pval=.
gen t2sitc=.


************************************************************
* Analysis
* Iterated FGLS
* Estimating product-by-product variant of equation (1) in text

* Loop over product categories
sort n_obs
forval i=1/`NN' {
	sum n_obs if sitcid==`i'
	local id=r(min)

	disp "`i' of `NN'"
	
	quietly {

	* Iteration 1: generate residuals
	areg g_imp wtem wtem_poor wpre wpre_poor yr* if sitcid==`i', absorb(cc_num) cluster(cc_num)
	predict double ehat, resid
	bys prdcnt_id: egen tmpv = sd(ehat)
	g double invvarehat = tmpv^(-2)
	sum invvarehat, detail

	* Iteration 2: FGLS round 1
	areg g_imp wtem wtem_poor wpre wpre_poor yr* [w=invvarehat] if sitcid==`i', absorb(cc_num) cluster(cc_num)
	predict double ehat2, resid
	bys prdcnt_id: egen tmpv2 = sd(ehat2)
	g double invvarehat2 = tmpv2^(-2)
	sum invvarehat2, detail
	
	* Iteration 3: FGLS round 2
	areg g_imp wtem wtem_poor wpre wpre_poor yr* [w=invvarehat2] if sitcid==`i', absorb(cc_num) cluster(cc_num)
	replace t2beta=_b[wtem_poor] if n_obs== `id'
	replace t2se=_se[wtem_poor] if n_obs== `id'
	lincom wtem + wtem_poor 
	replace t2beta=r(estimate) if n_obs== `id'
	replace t2se=r(se) if n_obs== `id'
	* attach sitc label to estimates
	replace t2sitc=sitc if n_obs== `id'
	
	drop tmpv ehat invvarehat tmpv2 ehat2 invvarehat2

	}
	
}
* End loop over product categories

replace t2tstat= t2beta / t2se
g absval = abs(t2tstat)
replace t2pval = ttail(t_obs,absval)*2
drop absval*

* Export results
sort t2tstat
list t2sitc t2beta t2se t2tstat t2pval if t2sitc!=.
outsheet t2sitc t2beta t2se t2tstat t2pval using Table2 if t2sitc!=., comma replace


